#if !defined( _DIRECT_LIGHTING_GLOBALS_FXH )
#define _DIRECT_LIGHTING_GLOBALS_FXH

//---------------------------------------------------------------------------------
//
// HLSL header file contains macros and definitions for specialisable direct lighting.
//


// Some shaders use a wraparound half-lambertian model for diffuse lighting. These defines
// control whether that is used or the more traditional lambertian model.

#if defined( _USE_HALF_LAMBERTIAN_LIGHTING_ )

#define CALCULATE_DIFFUSE_LIGHTING CalculateHalfLambertDiffuseLighting
#define SPECULAR_CUTOFF_LEVEL Rv( 0.5 )

#else

#define CALCULATE_DIFFUSE_LIGHTING CalculateDiffuseLighting
#define SPECULAR_CUTOFF_LEVEL Rv( 0.0 )

#endif


//---------------------------------------------------------------------------------
//
// 3DSMAX version:
//
// 1 unattenuated omni lightsource, position & colour linked to a scene light
//

#if defined( _3DSMAX_ )

#define DECLARE_DIRECT_LIGHTING_PARAMS			\
																						\
float3 lightPosition : POSITION							\
<																						\
	string UIName = "Light Position";					\
	string Object = "PointLight";							\
	string Space = "World";										\
	int RefID = 0;														\
	bool appEdit = false;											\
	bool export = false;											\
>;																					\
																						\
float4 lightColour : LightColor							\
<																						\
	string UIWidget = "None";									\
	string Object = "PointLight";							\
	int LightRef = 0;													\
	bool appEdit = false;											\
	bool export = false;											\
> = {1.0f, 1.0f, 1.0f, 1.0f};

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )				\
	/* N.B.	3dsmax doesn't like use of the token paste operator, */	\
	/* so this interpolator is hard coded at the maximum value.  */	\
	float3 lightVector : TEXCOORD9;

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS																\
	float3 worldLightVec = lightPosition - worldPos;												\
	_output.lightVector = worldLightVec;

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )				\
	{																													\
		float3 lightDirection0 = normalize( lightPosition - __worldPos__ );												\
		float4 lightColour0 = lightColour;																				\
		__accumulator__ += max( CALCULATE_DIFFUSE_LIGHTING( __worldNormal__, lightDirection0 ), 0.0f ) *  lightColour0;	\
	}

#if defined( USE_WARD_ANISOTROPIC_SPECULAR )

//
// Ward-style anisotropic lighting version
//
#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )												\
	float3 lightVector = normalize( _input.lightVector );																				\
	float diffuseLevel = CALCULATE_DIFFUSE_LIGHTING( normal, lightVector );												\
	__accumulator__ += diffuseTexColour * max( diffuseLevel, 0.0f ) * lightColour;							\
	if ( diffuseLevel > 0.0f )																																	\
	{																																														\
		__accumulator__ += specularTexColour *																										\
											 CalculateWardSpecularLighting( normal, perturbedUAxis, perturbedVAxis, alphaXValue, alphaYValue, lightVector, eye ) *	\
											 lightColour *																													\
											 globalSpecularFactorValue;																									\
	}

#elif defined( USE_TRANSMISSIVE_LIGHTING )

//
// Blinn+transmissive lighting version
//
#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )												\
	float3 lightVector = normalize( _input.lightVector );																				\
	float edotl = saturate( dot( eye, -lightVector ) );																					\
	edotl *= edotl;																																							\
	float diffuseLevel = CALCULATE_DIFFUSE_LIGHTING( normal, lightVector );												\
	float4 diffCol = diffuseTexColour * max( diffuseLevel, 0.0f ) * lightColour;								\
	float transLevel = edotl * saturate( dot( -normal, lightVector ) * 0.6f + 0.4f );						\
	float4 transCol = transmissiveTexColour * max( transLevel, 0.0f ) * lightColour * transmissiveness;	\
	__accumulator__ += diffCol + transCol;																											\
	if ( diffuseLevel > 0.0f )																																	\
	{																																														\
		float specularPower = lerp( minSpecPowerValue, maxSpecPowerValue, specularTexColour.a );						\
		__accumulator__ += specularTexColour *																										\
											 CalculateSpecularLighting( normal, lightVector, eye, specularPower ) *	\
											 lightColour *																					\
											 globalSpecularFactorValue;																									\
	}

#else

//
// Blinn-style lighting version
//
#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )												\
	float3 lightVector = normalize( _input.lightVector );																				\
	float diffuseLevel = CALCULATE_DIFFUSE_LIGHTING( normal, lightVector );						\
	__accumulator__ += diffuseTexColour * max( diffuseLevel, 0.0f ) * lightColour;						\
	if ( diffuseLevel > 0.0f )																																	\
	{																																														\
		float specularPower = lerp( minSpecPowerValue, maxSpecPowerValue, specularTexColour.a );						\
		__accumulator__ += specularTexColour *																										\
											 CalculateSpecularLighting( normal, lightVector, eye, specularPower ) *	\
											 lightColour *																					\
											 globalSpecularFactorValue;																									\
	}

#endif	// defined( USE_WARD_ANISOTROPIC_SPECULAR )

#else

//---------------------------------------------------------------------------------
//
// Engine versions:
//
// Exact implementation depends on the combination of light sources defined in the
// compiler commandline.
//
// Currently defined: Any combination from 1-3 lights
// Faceted define code ::= _LIGHTING_[<light>][<light>][<light>]_
// <light> ::= <type>
// <type> ::= D|P|S        ( Directional / Point / Spot )
//
// (lights are always sorted in order DPS, to remove duplicate combinations)
//
// Shadowing faceted define code ::= _SHADOWS_<shadow><shadow><shadow>_
// <shadow> ::= Y|N				(Yes, shadowed / No, unshadowed)
//

//
// Common macros
//
#include "directionallights.fxh"
#include "pointlights.fxh"
#include "spotlights.fxh"
#include "shadow_globals.fxh"

//
// Macros common to all lights & combinations
//

#define DO_PS_LIGHTING_SETUP	\
	LPFLOAT PSL_attenuation;

#if defined( _LIGHTING_PRECALCULATED_ )

// Precalculated lighting doesn't need any pixel shader lighting,
// so setting this value to 0.0 should kill it off (assuming the
// rest of the attenuation calcs are nullified too).
#define DO_PS_LIGHT_CALCULATIONS	\
	PSL_attenuation = Rv( 0.0 );

#else

#define DO_PS_LIGHT_CALCULATIONS	\
	PSL_attenuation = Rv( 1.0 );

#endif


//
// Shadowing combinatorial macros
//
// To avoid having a set of macros for every combination of shadows & lights, these
// meta-macros are defined for each shadow combination, which will in turn generate
// the correct macro declarations (or nothing) later on when the light types are known.
//
//

#if defined( _SHADOWS_NNN_ )

#if defined(_LIGHTING_DYNAMIC_)
	#define NEED_SHADOW_INFO
#endif

#define FIRST_LIGHT_DIR_SHADOW_PARAMS
#define FIRST_LIGHT_POINT_SHADOW_PARAMS
#define FIRST_LIGHT_SPOT_SHADOW_PARAMS

#define FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define FIRST_LIGHT_DIR_VS_SHADOW_CALC
#define FIRST_LIGHT_POINT_VS_SHADOW_CALC
#define FIRST_LIGHT_SPOT_VS_SHADOW_CALC

#define FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#define SECOND_LIGHT_DIR_SHADOW_PARAMS
#define SECOND_LIGHT_POINT_SHADOW_PARAMS
#define SECOND_LIGHT_SPOT_SHADOW_PARAMS

#define SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define SECOND_LIGHT_DIR_VS_SHADOW_CALC
#define SECOND_LIGHT_POINT_VS_SHADOW_CALC
#define SECOND_LIGHT_SPOT_VS_SHADOW_CALC

#define SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#define THIRD_LIGHT_DIR_SHADOW_PARAMS
#define THIRD_LIGHT_POINT_SHADOW_PARAMS
#define THIRD_LIGHT_SPOT_SHADOW_PARAMS

#define THIRD_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define THIRD_LIGHT_DIR_VS_SHADOW_CALC
#define THIRD_LIGHT_POINT_VS_SHADOW_CALC
#define THIRD_LIGHT_SPOT_VS_SHADOW_CALC

#define THIRD_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#elif defined( _SHADOWS_YNN_ )

#define NEED_WORLDPOS

#if defined(_LIGHTING_DYNAMIC_)
	#define NEED_SHADOW_INFO
#endif

#define FIRST_LIGHT_DIR_SHADOW_PARAMS	DECLARE_DIRECTIONAL_SHADOW_PARAMS( 0 )
#define FIRST_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 0 )
#define FIRST_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 0 )

#define FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 0 )
#define FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 0 )
#define FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 0 )

#define FIRST_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 0 )
#define FIRST_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 0 )
#define FIRST_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 0 )

#define FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 0, __eye__ )
#define FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 0, __eye__ )
#define FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 0, __eye__ )

#define SECOND_LIGHT_DIR_SHADOW_PARAMS
#define SECOND_LIGHT_POINT_SHADOW_PARAMS
#define SECOND_LIGHT_SPOT_SHADOW_PARAMS

#define SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define SECOND_LIGHT_DIR_VS_SHADOW_CALC
#define SECOND_LIGHT_POINT_VS_SHADOW_CALC
#define SECOND_LIGHT_SPOT_VS_SHADOW_CALC

#define SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#define THIRD_LIGHT_DIR_SHADOW_PARAMS
#define THIRD_LIGHT_POINT_SHADOW_PARAMS
#define THIRD_LIGHT_SPOT_SHADOW_PARAMS

#define THIRD_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define THIRD_LIGHT_DIR_VS_SHADOW_CALC
#define THIRD_LIGHT_POINT_VS_SHADOW_CALC
#define THIRD_LIGHT_SPOT_VS_SHADOW_CALC

#define THIRD_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#elif defined( _SHADOWS_NYN_ )

#define NEED_WORLDPOS

#define FIRST_LIGHT_DIR_SHADOW_PARAMS
#define FIRST_LIGHT_POINT_SHADOW_PARAMS
#define FIRST_LIGHT_SPOT_SHADOW_PARAMS

#define FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define FIRST_LIGHT_DIR_VS_SHADOW_CALC
#define FIRST_LIGHT_POINT_VS_SHADOW_CALC
#define FIRST_LIGHT_SPOT_VS_SHADOW_CALC

#define FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#define SECOND_LIGHT_DIR_SHADOW_PARAMS DECLARE_DIRECTIONAL_SHADOW_PARAMS( 0 )
#define SECOND_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 0 )
#define SECOND_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 0 )

#define SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 0 )
#define SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 0 )
#define SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 0 )

#define SECOND_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 0 )
#define SECOND_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 0 )
#define SECOND_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 0 )

#define SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 0, __eye__ )
#define SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 0, __eye__ )
#define SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 0, __eye__ )

#define THIRD_LIGHT_DIR_SHADOW_PARAMS
#define THIRD_LIGHT_POINT_SHADOW_PARAMS
#define THIRD_LIGHT_SPOT_SHADOW_PARAMS

#define THIRD_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define THIRD_LIGHT_DIR_VS_SHADOW_CALC
#define THIRD_LIGHT_POINT_VS_SHADOW_CALC
#define THIRD_LIGHT_SPOT_VS_SHADOW_CALC

#define THIRD_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#elif defined( _SHADOWS_YYN_ )

#define NEED_WORLDPOS

#define FIRST_LIGHT_DIR_SHADOW_PARAMS	DECLARE_DIRECTIONAL_SHADOW_PARAMS( 0 )
#define FIRST_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 0 )
#define FIRST_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 0 )

#define FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 0 )
#define FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 0 )
#define FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 0 )

#define FIRST_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 0 )
#define FIRST_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 0 )
#define FIRST_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 0 )

#define FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 0, __eye__ )
#define FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 0, __eye__ )
#define FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 0, __eye__ )

#define SECOND_LIGHT_DIR_SHADOW_PARAMS DECLARE_DIRECTIONAL_SHADOW_PARAMS( 1 )
#define SECOND_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 1 )
#define SECOND_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 1 )

#define SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 1 )
#define SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 1 )
#define SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 1 )

#define SECOND_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 1 )
#define SECOND_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 1 )
#define SECOND_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 1 )

#define SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 1, __eye__ )
#define SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 1, __eye__ )
#define SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 1, __eye__ )

#define THIRD_LIGHT_DIR_SHADOW_PARAMS
#define THIRD_LIGHT_POINT_SHADOW_PARAMS
#define THIRD_LIGHT_SPOT_SHADOW_PARAMS

#define THIRD_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define THIRD_LIGHT_DIR_VS_SHADOW_CALC
#define THIRD_LIGHT_POINT_VS_SHADOW_CALC
#define THIRD_LIGHT_SPOT_VS_SHADOW_CALC

#define THIRD_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#elif defined( _SHADOWS_NNY_ )

#define NEED_WORLDPOS

#define FIRST_LIGHT_DIR_SHADOW_PARAMS
#define FIRST_LIGHT_POINT_SHADOW_PARAMS
#define FIRST_LIGHT_SPOT_SHADOW_PARAMS

#define FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define FIRST_LIGHT_DIR_VS_SHADOW_CALC
#define FIRST_LIGHT_POINT_VS_SHADOW_CALC
#define FIRST_LIGHT_SPOT_VS_SHADOW_CALC

#define FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#define SECOND_LIGHT_DIR_SHADOW_PARAMS
#define SECOND_LIGHT_POINT_SHADOW_PARAMS
#define SECOND_LIGHT_SPOT_SHADOW_PARAMS

#define SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define SECOND_LIGHT_DIR_VS_SHADOW_CALC
#define SECOND_LIGHT_POINT_VS_SHADOW_CALC
#define SECOND_LIGHT_SPOT_VS_SHADOW_CALC

#define SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#define THIRD_LIGHT_DIR_SHADOW_PARAMS DECLARE_DIRECTIONAL_SHADOW_PARAMS( 0 )
#define THIRD_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 0 )
#define THIRD_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 0 )

#define THIRD_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 0 )
#define THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 0 )
#define THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 0 )

#define THIRD_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 0 )
#define THIRD_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 0 )
#define THIRD_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 0 )

#define THIRD_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 0, __eye__ )
#define THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 0, __eye__ )
#define THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 0, __eye__ )

#elif defined( _SHADOWS_YNY_ )

#define NEED_WORLDPOS

#define FIRST_LIGHT_DIR_SHADOW_PARAMS	DECLARE_DIRECTIONAL_SHADOW_PARAMS( 0 )
#define FIRST_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 0 )
#define FIRST_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 0 )

#define FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 0 )
#define FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 0 )
#define FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 0 )

#define FIRST_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 0 )
#define FIRST_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 0 )
#define FIRST_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 0 )

#define FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 0, __eye__ )
#define FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 0, __eye__ )
#define FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 0, __eye__ )

#define SECOND_LIGHT_DIR_SHADOW_PARAMS
#define SECOND_LIGHT_POINT_SHADOW_PARAMS
#define SECOND_LIGHT_SPOT_SHADOW_PARAMS

#define SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define SECOND_LIGHT_DIR_VS_SHADOW_CALC
#define SECOND_LIGHT_POINT_VS_SHADOW_CALC
#define SECOND_LIGHT_SPOT_VS_SHADOW_CALC

#define SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#define THIRD_LIGHT_DIR_SHADOW_PARAMS DECLARE_DIRECTIONAL_SHADOW_PARAMS( 1 )
#define THIRD_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 1 )
#define THIRD_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 1 )

#define THIRD_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 1 )
#define THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 1 )
#define THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 1 )

#define THIRD_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 1 )
#define THIRD_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 1 )
#define THIRD_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 1 )

#define THIRD_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 1, __eye__ )
#define THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 1, __eye__ )
#define THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 1, __eye__ )

#elif defined( _SHADOWS_NYY_ )

#define NEED_WORLDPOS

#define FIRST_LIGHT_DIR_SHADOW_PARAMS
#define FIRST_LIGHT_POINT_SHADOW_PARAMS
#define FIRST_LIGHT_SPOT_SHADOW_PARAMS

#define FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )
#define FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )
#define FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define FIRST_LIGHT_DIR_VS_SHADOW_CALC
#define FIRST_LIGHT_POINT_VS_SHADOW_CALC
#define FIRST_LIGHT_SPOT_VS_SHADOW_CALC

#define FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )
#define FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )
#define FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )

#define SECOND_LIGHT_DIR_SHADOW_PARAMS DECLARE_DIRECTIONAL_SHADOW_PARAMS( 0 )
#define SECOND_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 0 )
#define SECOND_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 0 )

#define SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 0 )
#define SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 0 )
#define SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 0 )

#define SECOND_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 0 )
#define SECOND_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 0 )
#define SECOND_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 0 )

#define SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 0, __eye__ )
#define SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 0, __eye__ )
#define SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 0, __eye__ )

#define THIRD_LIGHT_DIR_SHADOW_PARAMS DECLARE_DIRECTIONAL_SHADOW_PARAMS( 1 )
#define THIRD_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 1 )
#define THIRD_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 1 )

#define THIRD_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 1 )
#define THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 1 )
#define THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 1 )

#define THIRD_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 1 )
#define THIRD_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 1 )
#define THIRD_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 1 )

#define THIRD_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 1, __eye__ )
#define THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 1, __eye__ )
#define THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 1, __eye__ )

#elif defined( _SHADOWS_YYY_ )

#define NEED_WORLDPOS

#define FIRST_LIGHT_DIR_SHADOW_PARAMS	DECLARE_DIRECTIONAL_SHADOW_PARAMS( 0 )
#define FIRST_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 0 )
#define FIRST_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 0 )

#define FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 0 )
#define FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 0 )
#define FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 0 )

#define FIRST_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 0 )
#define FIRST_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 0 )
#define FIRST_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 0 )

#define FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 0, __eye__ )
#define FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 0, __eye__ )
#define FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 0, __eye__ )

#define SECOND_LIGHT_DIR_SHADOW_PARAMS DECLARE_DIRECTIONAL_SHADOW_PARAMS( 1 )
#define SECOND_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 1 )
#define SECOND_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 1 )

#define SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 1 )
#define SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 1 )
#define SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 1 )

#define SECOND_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 1 )
#define SECOND_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 1 )
#define SECOND_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 1 )

#define SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 1, __eye__ )
#define SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 1, __eye__ )
#define SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 1, __eye__ )

#define THIRD_LIGHT_DIR_SHADOW_PARAMS DECLARE_DIRECTIONAL_SHADOW_PARAMS( 2 )
#define THIRD_LIGHT_POINT_SHADOW_PARAMS DECLARE_POINT_SHADOW_PARAMS( 2 )
#define THIRD_LIGHT_SPOT_SHADOW_PARAMS DECLARE_SPOT_SHADOW_PARAMS( 2 )

#define THIRD_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ ) DECLARE_DIRECTIONAL_SHADOW_INTERPOLATORS( __index__, 2 )
#define THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_POINT_SHADOW_INTERPOLATORS( __index__, 2 )
#define THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ ) DECLARE_SPOT_SHADOW_INTERPOLATORS( __index__, 2 )

#define THIRD_LIGHT_DIR_VS_SHADOW_CALC DO_VS_DIRECTIONAL_SHADOW_CALCULATIONS( 2 )
#define THIRD_LIGHT_POINT_VS_SHADOW_CALC DO_VS_POINT_SHADOW_CALCULATIONS( 2 )
#define THIRD_LIGHT_SPOT_VS_SHADOW_CALC DO_VS_SPOT_SHADOW_CALCULATIONS( 2 )

#define THIRD_LIGHT_DIR_PS_SHADOW_CALC( __eye__ ) DO_PS_DIRECTIONAL_SHADOW_CALCULATIONS( 2, __eye__ )
#define THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ ) DO_PS_POINT_SHADOW_CALCULATIONS( 2, __eye__ )
#define THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ ) DO_PS_SPOT_SHADOW_CALCULATIONS( 2, __eye__ )

#endif // defined( _SHADOWS_NNN_ )


#if defined( _LIGHTING_D_ )

//
// One directional light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_DIRECTIONAL_PARAMS( 0 )				\
	FIRST_LIGHT_DIR_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 0 )					\
	FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS			\
	DO_VS_DIRECTIONAL_LIGHTING_SETUP							\
	DO_VS_DIRECTIONAL_SHADOW_SETUP								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 0 )	\
	FIRST_LIGHT_DIR_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )		\
	DO_PS_LIGHTING_SETUP																					\
	DO_PS_DIRECTIONAL_LIGHTING_SETUP															\
	DO_PS_DIRECTIONAL_SHADOW_SETUP																\
	DO_PS_LIGHT_CALCULATIONS																			\
	FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 0 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )				\
	DO_VERTEXLIGHTING_DIRECTIONAL_LIGHTING_CALCULATIONS( __worldNormal__, __accumulator__, 0 )


#elif defined( _LIGHTING_DD_ )

//
// Two directional lights
//

#define DECLARE_DIRECT_LIGHTING_PARAMS		\
	DECLARE_DIRECTIONAL_PARAMS( 0 )					\
	DECLARE_DIRECTIONAL_PARAMS( 1 )					\
	FIRST_LIGHT_DIR_SHADOW_PARAMS						\
	SECOND_LIGHT_DIR_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 0 )					\
	FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )					\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 1 )					\
	SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS			\
	DO_VS_DIRECTIONAL_LIGHTING_SETUP							\
	DO_VS_DIRECTIONAL_SHADOW_SETUP								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 0 )	\
	FIRST_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 1 )	\
	SECOND_LIGHT_DIR_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )		\
	DO_PS_LIGHTING_SETUP																					\
	DO_PS_DIRECTIONAL_LIGHTING_SETUP															\
	DO_PS_DIRECTIONAL_SHADOW_SETUP																\
	DO_PS_LIGHT_CALCULATIONS																			\
	FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 0 )	\
	DO_PS_LIGHT_CALCULATIONS																			\
	SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 1 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )				\
	DO_VERTEXLIGHTING_DIRECTIONAL_LIGHTING_CALCULATIONS( __worldNormal__, __accumulator__, 0 )			\
	DO_VERTEXLIGHTING_DIRECTIONAL_LIGHTING_CALCULATIONS( __worldNormal__, __accumulator__, 1 )


#elif defined( _LIGHTING_DDD_ )

//
// Three directional lights
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_DIRECTIONAL_PARAMS( 0 )				\
	DECLARE_DIRECTIONAL_PARAMS( 1 )				\
	DECLARE_DIRECTIONAL_PARAMS( 2 )				\
	FIRST_LIGHT_DIR_SHADOW_PARAMS					\
	SECOND_LIGHT_DIR_SHADOW_PARAMS				\
	THIRD_LIGHT_DIR_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 0 )					\
	FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )					\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 1 )					\
	SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 2 )					\
	THIRD_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS			\
	DO_VS_DIRECTIONAL_LIGHTING_SETUP							\
	DO_VS_DIRECTIONAL_SHADOW_SETUP								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 0 )	\
	FIRST_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 1 )	\
	SECOND_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 2 )	\
	THIRD_LIGHT_DIR_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )		\
	DO_PS_LIGHTING_SETUP																					\
	DO_PS_DIRECTIONAL_LIGHTING_SETUP															\
	DO_PS_DIRECTIONAL_SHADOW_SETUP																\
	DO_PS_LIGHT_CALCULATIONS																			\
	FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 0 )	\
	DO_PS_LIGHT_CALCULATIONS																			\
	SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 1 )	\
	DO_PS_LIGHT_CALCULATIONS																			\
	THIRD_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 2 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )				\
	DO_VERTEXLIGHTING_DIRECTIONAL_LIGHTING_CALCULATIONS( __worldNormal__, __accumulator__, 0 )			\
	DO_VERTEXLIGHTING_DIRECTIONAL_LIGHTING_CALCULATIONS( __worldNormal__, __accumulator__, 1 )			\
	DO_VERTEXLIGHTING_DIRECTIONAL_LIGHTING_CALCULATIONS( __worldNormal__, __accumulator__, 2 )


#elif defined( _LIGHTING_P_ )

//
// One point light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_POINT_PARAMS( 0 )							\
	FIRST_LIGHT_POINT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_POINT_INTERPOLATORS( __index__, 0 )								\
	FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS	\
	DO_VS_POINT_LIGHTING_SETUP								\
	DO_VS_POINT_SHADOW_SETUP									\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 0 )		\
	FIRST_LIGHT_POINT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )	\
	DO_PS_LIGHTING_SETUP																				\
	DO_PS_POINT_LIGHTING_SETUP																	\
	DO_PS_POINT_SHADOW_SETUP																		\
	DO_PS_LIGHT_CALCULATIONS																		\
	FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 0 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_PP_ )

//
// Two point lights
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_POINT_PARAMS( 0 )							\
	DECLARE_POINT_PARAMS( 1 )							\
	FIRST_LIGHT_POINT_SHADOW_PARAMS				\
	SECOND_LIGHT_POINT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_POINT_INTERPOLATORS( __index__, 0 )								\
	FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_POINT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS	\
	DO_VS_POINT_LIGHTING_SETUP								\
	DO_VS_POINT_SHADOW_SETUP									\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 0 )		\
	FIRST_LIGHT_POINT_VS_SHADOW_CALC					\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 1 )		\
	SECOND_LIGHT_POINT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )	\
	DO_PS_LIGHTING_SETUP																				\
	DO_PS_POINT_LIGHTING_SETUP																	\
	DO_PS_POINT_SHADOW_SETUP																		\
	DO_PS_LIGHT_CALCULATIONS																		\
	FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 0 )			\
	DO_PS_LIGHT_CALCULATIONS																		\
	SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 1 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_PPP_ )

//
// Three point lights
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_POINT_PARAMS( 0 )							\
	DECLARE_POINT_PARAMS( 1 )							\
	DECLARE_POINT_PARAMS( 2 )							\
	FIRST_LIGHT_POINT_SHADOW_PARAMS				\
	SECOND_LIGHT_POINT_SHADOW_PARAMS			\
	THIRD_LIGHT_POINT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_POINT_INTERPOLATORS( __index__, 0 )								\
	FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_POINT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )			\
	DECLARE_POINT_INTERPOLATORS( __index__, 2 )								\
	THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS	\
	DO_VS_POINT_LIGHTING_SETUP								\
	DO_VS_POINT_SHADOW_SETUP									\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 0 )		\
	FIRST_LIGHT_POINT_VS_SHADOW_CALC					\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 1 )		\
	SECOND_LIGHT_POINT_VS_SHADOW_CALC					\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 2 )		\
	THIRD_LIGHT_POINT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )	\
	DO_PS_LIGHTING_SETUP																				\
	DO_PS_POINT_LIGHTING_SETUP																	\
	DO_PS_POINT_SHADOW_SETUP																		\
	DO_PS_LIGHT_CALCULATIONS																		\
	FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 0 )			\
	DO_PS_LIGHT_CALCULATIONS																		\
	SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 1 )			\
	DO_PS_LIGHT_CALCULATIONS																		\
	THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 2 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_S_ )

//
// One spot light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_SPOT_PARAMS( 0 )							\
	FIRST_LIGHT_SPOT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_SPOT_INTERPOLATORS( __index__, 0 )								\
	FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS	\
	DO_VS_SPOT_LIGHTING_SETUP									\
	DO_VS_SPOT_SHADOW_SETUP										\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 0 )			\
	FIRST_LIGHT_SPOT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )	\
	DO_PS_LIGHTING_SETUP																				\
	DO_PS_SPOT_LIGHTING_SETUP																		\
	DO_PS_SPOT_SHADOW_SETUP																			\
	DO_PS_LIGHT_CALCULATIONS																		\
	FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 0 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_SS_ )

//
// Two spot lights
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_SPOT_PARAMS( 0 )							\
	DECLARE_SPOT_PARAMS( 1 )							\
	FIRST_LIGHT_SPOT_SHADOW_PARAMS				\
	SECOND_LIGHT_SPOT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_SPOT_INTERPOLATORS( __index__, 0 )								\
	FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_SPOT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS	\
	DO_VS_SPOT_LIGHTING_SETUP									\
	DO_VS_SPOT_SHADOW_SETUP										\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 0 )			\
	FIRST_LIGHT_SPOT_VS_SHADOW_CALC						\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 1 )			\
	SECOND_LIGHT_SPOT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )	\
	DO_PS_LIGHTING_SETUP																				\
	DO_PS_SPOT_LIGHTING_SETUP																		\
	DO_PS_SPOT_SHADOW_SETUP																			\
	DO_PS_LIGHT_CALCULATIONS																		\
	FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 0 )			\
	DO_PS_LIGHT_CALCULATIONS																		\
	SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 1 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_SSS_ )

//
// Three spot lights
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_SPOT_PARAMS( 0 )							\
	DECLARE_SPOT_PARAMS( 1 )							\
	DECLARE_SPOT_PARAMS( 2 )							\
	FIRST_LIGHT_SPOT_SHADOW_PARAMS				\
	SECOND_LIGHT_SPOT_SHADOW_PARAMS				\
	THIRD_LIGHT_SPOT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_SPOT_INTERPOLATORS( __index__, 0 )								\
	FIRST_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_SPOT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_SPOT_INTERPOLATORS( __index__, 2 )								\
	THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS	\
	DO_VS_SPOT_LIGHTING_SETUP									\
	DO_VS_SPOT_SHADOW_SETUP										\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 0 )			\
	FIRST_LIGHT_SPOT_VS_SHADOW_CALC						\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 1 )			\
	SECOND_LIGHT_SPOT_VS_SHADOW_CALC					\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 2 )			\
	THIRD_LIGHT_SPOT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )	\
	DO_PS_LIGHTING_SETUP																				\
	DO_PS_SPOT_LIGHTING_SETUP																		\
	DO_PS_SPOT_SHADOW_SETUP																			\
	DO_PS_LIGHT_CALCULATIONS																		\
	FIRST_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 0 )			\
	DO_PS_LIGHT_CALCULATIONS																		\
	SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 1 )			\
	DO_PS_LIGHT_CALCULATIONS																		\
	THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 2 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_DP_ )

//
// One directional + one point light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_DIRECTIONAL_PARAMS( 0 )				\
	DECLARE_POINT_PARAMS( 1 )							\
	FIRST_LIGHT_DIR_SHADOW_PARAMS					\
	SECOND_LIGHT_POINT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 0 )					\
	FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )					\
	DECLARE_POINT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS			\
	DO_VS_DIRECTIONAL_LIGHTING_SETUP							\
	DO_VS_DIRECTIONAL_SHADOW_SETUP								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 0 )	\
	FIRST_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_POINT_LIGHTING_SETUP										\
	DO_VS_POINT_SHADOW_SETUP											\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 1 )				\
	SECOND_LIGHT_POINT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )		\
	DO_PS_LIGHTING_SETUP																					\
	DO_PS_DIRECTIONAL_LIGHTING_SETUP															\
	DO_PS_DIRECTIONAL_SHADOW_SETUP																\
	DO_PS_LIGHT_CALCULATIONS																			\
	FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 0 )	\
	DO_PS_POINT_LIGHTING_SETUP																		\
	DO_PS_POINT_SHADOW_SETUP																			\
	DO_PS_LIGHT_CALCULATIONS																			\
	SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 1 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_DS_ )

//
// One directional + one spot light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_DIRECTIONAL_PARAMS( 0 )				\
	DECLARE_SPOT_PARAMS( 1 )							\
	FIRST_LIGHT_DIR_SHADOW_PARAMS					\
	SECOND_LIGHT_SPOT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 0 )					\
	FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )					\
	DECLARE_SPOT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS			\
	DO_VS_DIRECTIONAL_LIGHTING_SETUP							\
	DO_VS_DIRECTIONAL_SHADOW_SETUP								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 0 )	\
	FIRST_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_SPOT_LIGHTING_SETUP											\
	DO_VS_SPOT_SHADOW_SETUP												\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 1 )					\
	SECOND_LIGHT_SPOT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )		\
	DO_PS_LIGHTING_SETUP																					\
	DO_PS_DIRECTIONAL_LIGHTING_SETUP															\
	DO_PS_DIRECTIONAL_SHADOW_SETUP																\
	DO_PS_LIGHT_CALCULATIONS																			\
	FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 0 )	\
	DO_PS_SPOT_LIGHTING_SETUP																			\
	DO_PS_SPOT_SHADOW_SETUP																				\
	DO_PS_LIGHT_CALCULATIONS																			\
	SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 1 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_DPS_ )

//
// Directional + point + spot light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_DIRECTIONAL_PARAMS( 0 )				\
	DECLARE_POINT_PARAMS( 1 )							\
	DECLARE_SPOT_PARAMS( 2 )							\
	FIRST_LIGHT_DIR_SHADOW_PARAMS					\
	SECOND_LIGHT_POINT_SHADOW_PARAMS			\
	THIRD_LIGHT_SPOT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 0 )					\
	FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )					\
	DECLARE_POINT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )			\
	DECLARE_SPOT_INTERPOLATORS( __index__, 2 )								\
	THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS			\
	DO_VS_DIRECTIONAL_LIGHTING_SETUP							\
	DO_VS_DIRECTIONAL_SHADOW_SETUP								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 0 )	\
	FIRST_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_POINT_LIGHTING_SETUP										\
	DO_VS_POINT_SHADOW_SETUP											\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 1 )				\
	SECOND_LIGHT_POINT_VS_SHADOW_CALC							\
	DO_VS_SPOT_LIGHTING_SETUP											\
	DO_VS_SPOT_SHADOW_SETUP												\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 2 )					\
	THIRD_LIGHT_SPOT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )		\
	DO_PS_LIGHTING_SETUP																					\
	DO_PS_DIRECTIONAL_LIGHTING_SETUP															\
	DO_PS_DIRECTIONAL_SHADOW_SETUP																\
	DO_PS_LIGHT_CALCULATIONS																			\
	FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 0 )	\
	DO_PS_POINT_LIGHTING_SETUP																		\
	DO_PS_POINT_SHADOW_SETUP																			\
	DO_PS_LIGHT_CALCULATIONS																			\
	SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 1 )				\
	DO_PS_SPOT_LIGHTING_SETUP																			\
	DO_PS_SPOT_SHADOW_SETUP																				\
	DO_PS_LIGHT_CALCULATIONS																			\
	THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 2 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_DPP_ )

//
// Directional + point + point light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_DIRECTIONAL_PARAMS( 0 )				\
	DECLARE_POINT_PARAMS( 1 )							\
	DECLARE_POINT_PARAMS( 2 )							\
	FIRST_LIGHT_DIR_SHADOW_PARAMS					\
	SECOND_LIGHT_POINT_SHADOW_PARAMS			\
	THIRD_LIGHT_POINT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 0 )					\
	FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )					\
	DECLARE_POINT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )			\
	DECLARE_POINT_INTERPOLATORS( __index__, 2 )								\
	THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS			\
	DO_VS_DIRECTIONAL_LIGHTING_SETUP							\
	DO_VS_DIRECTIONAL_SHADOW_SETUP								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 0 )	\
	FIRST_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_POINT_LIGHTING_SETUP										\
	DO_VS_POINT_SHADOW_SETUP											\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 1 )				\
	SECOND_LIGHT_POINT_VS_SHADOW_CALC							\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 2 )				\
	THIRD_LIGHT_POINT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )		\
	DO_PS_LIGHTING_SETUP																					\
	DO_PS_DIRECTIONAL_LIGHTING_SETUP															\
	DO_PS_DIRECTIONAL_SHADOW_SETUP																\
	DO_PS_LIGHT_CALCULATIONS																			\
	FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 0 )	\
	DO_PS_POINT_LIGHTING_SETUP																		\
	DO_PS_POINT_SHADOW_SETUP																			\
	DO_PS_LIGHT_CALCULATIONS																			\
	SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 1 )				\
	DO_PS_LIGHT_CALCULATIONS																			\
	THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 2 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_DSS_ )

//
// Directional + spot + spot light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_DIRECTIONAL_PARAMS( 0 )				\
	DECLARE_SPOT_PARAMS( 1 )							\
	DECLARE_SPOT_PARAMS( 2 )							\
	FIRST_LIGHT_DIR_SHADOW_PARAMS					\
	SECOND_LIGHT_SPOT_SHADOW_PARAMS				\
	THIRD_LIGHT_SPOT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 0 )					\
	FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )					\
	DECLARE_SPOT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_SPOT_INTERPOLATORS( __index__, 2 )								\
	THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS			\
	DO_VS_DIRECTIONAL_LIGHTING_SETUP							\
	DO_VS_DIRECTIONAL_SHADOW_SETUP								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 0 )	\
	FIRST_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_SPOT_LIGHTING_SETUP											\
	DO_VS_SPOT_SHADOW_SETUP												\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 1 )					\
	SECOND_LIGHT_SPOT_VS_SHADOW_CALC							\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 2 )					\
	THIRD_LIGHT_SPOT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )		\
	DO_PS_LIGHTING_SETUP																					\
	DO_PS_DIRECTIONAL_LIGHTING_SETUP															\
	DO_PS_DIRECTIONAL_SHADOW_SETUP																\
	DO_PS_LIGHT_CALCULATIONS																			\
	FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 0 )	\
	DO_PS_SPOT_LIGHTING_SETUP																			\
	DO_PS_SPOT_SHADOW_SETUP																				\
	DO_PS_LIGHT_CALCULATIONS																			\
	SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 1 )				\
	DO_PS_LIGHT_CALCULATIONS																			\
	THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 2 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_DDP_ )

//
// Directional + directional + point light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_DIRECTIONAL_PARAMS( 0 )				\
	DECLARE_DIRECTIONAL_PARAMS( 1 )				\
	DECLARE_POINT_PARAMS( 2 )							\
	FIRST_LIGHT_DIR_SHADOW_PARAMS					\
	SECOND_LIGHT_DIR_SHADOW_PARAMS				\
	THIRD_LIGHT_POINT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 0 )					\
	FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )					\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 1 )					\
	SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_POINT_INTERPOLATORS( __index__, 2 )								\
	THIRD_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS			\
	DO_VS_DIRECTIONAL_LIGHTING_SETUP							\
	DO_VS_DIRECTIONAL_SHADOW_SETUP								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 0 )	\
	FIRST_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 1 )	\
	SECOND_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_POINT_LIGHTING_SETUP										\
	DO_VS_POINT_SHADOW_SETUP											\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 2 )				\
	THIRD_LIGHT_POINT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )		\
	DO_PS_LIGHTING_SETUP																					\
	DO_PS_DIRECTIONAL_LIGHTING_SETUP															\
	DO_PS_DIRECTIONAL_SHADOW_SETUP																\
	DO_PS_LIGHT_CALCULATIONS																			\
	FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 0 )	\
	DO_PS_LIGHT_CALCULATIONS																			\
	SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 1 )	\
	DO_PS_POINT_LIGHTING_SETUP																		\
	DO_PS_POINT_SHADOW_SETUP																			\
	DO_PS_LIGHT_CALCULATIONS																			\
	THIRD_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 2 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_DDS_ )

//
// Directional + directional + spot light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_DIRECTIONAL_PARAMS( 0 )				\
	DECLARE_DIRECTIONAL_PARAMS( 1 )				\
	DECLARE_SPOT_PARAMS( 2 )							\
	FIRST_LIGHT_DIR_SHADOW_PARAMS					\
	SECOND_LIGHT_DIR_SHADOW_PARAMS				\
	THIRD_LIGHT_SPOT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 0 )					\
	FIRST_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )					\
	DECLARE_DIRECTIONAL_INTERPOLATORS( __index__, 1 )					\
	SECOND_LIGHT_DIR_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_SPOT_INTERPOLATORS( __index__, 2 )								\
	THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS			\
	DO_VS_DIRECTIONAL_LIGHTING_SETUP							\
	DO_VS_DIRECTIONAL_SHADOW_SETUP								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 0 )	\
	FIRST_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_DIRECTIONAL_LIGHTING_CALCULATIONS( 1 )	\
	SECOND_LIGHT_DIR_VS_SHADOW_CALC								\
	DO_VS_SPOT_LIGHTING_SETUP											\
	DO_VS_SPOT_SHADOW_SETUP												\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 2 )					\
	THIRD_LIGHT_SPOT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )		\
	DO_PS_LIGHTING_SETUP																					\
	DO_PS_DIRECTIONAL_LIGHTING_SETUP															\
	DO_PS_DIRECTIONAL_SHADOW_SETUP																\
	DO_PS_LIGHT_CALCULATIONS																			\
	FIRST_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 0 )	\
	DO_PS_LIGHT_CALCULATIONS																			\
	SECOND_LIGHT_DIR_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_DIRECTIONAL_LIGHTING_CALCULATIONS( __accumulator__, 1 )	\
	DO_PS_SPOT_LIGHTING_SETUP																			\
	DO_PS_SPOT_SHADOW_SETUP																				\
	DO_PS_LIGHT_CALCULATIONS																			\
	THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )																\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 2 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_PS_ )

//
// One point + one spot light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_POINT_PARAMS( 0 )							\
	DECLARE_SPOT_PARAMS( 1 )							\
	FIRST_LIGHT_POINT_SHADOW_PARAMS				\
	SECOND_LIGHT_SPOT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_POINT_INTERPOLATORS( __index__, 0 )								\
	FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_SPOT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS	\
	DO_VS_POINT_LIGHTING_SETUP								\
	DO_VS_POINT_SHADOW_SETUP									\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 0 )		\
	FIRST_LIGHT_POINT_VS_SHADOW_CALC					\
	DO_VS_SPOT_LIGHTING_SETUP									\
	DO_VS_SPOT_SHADOW_SETUP										\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 1 )			\
	SECOND_LIGHT_SPOT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )	\
	DO_PS_LIGHTING_SETUP																				\
	DO_PS_POINT_LIGHTING_SETUP																	\
	DO_PS_POINT_SHADOW_SETUP																		\
	DO_PS_LIGHT_CALCULATIONS																		\
	FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 0 )			\
	DO_PS_SPOT_LIGHTING_SETUP																		\
	DO_PS_SPOT_SHADOW_SETUP																			\
	DO_PS_LIGHT_CALCULATIONS																		\
	SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 1 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_PSS_ )

//
// One point + two spot lights
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_POINT_PARAMS( 0 )							\
	DECLARE_SPOT_PARAMS( 1 )							\
	DECLARE_SPOT_PARAMS( 2 )							\
	FIRST_LIGHT_POINT_SHADOW_PARAMS				\
	SECOND_LIGHT_SPOT_SHADOW_PARAMS				\
	THIRD_LIGHT_SPOT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_POINT_INTERPOLATORS( __index__, 0 )								\
	FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_SPOT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_SPOT_INTERPOLATORS( __index__, 2 )								\
	THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS	\
	DO_VS_POINT_LIGHTING_SETUP								\
	DO_VS_POINT_SHADOW_SETUP									\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 0 )		\
	FIRST_LIGHT_POINT_VS_SHADOW_CALC					\
	DO_VS_SPOT_LIGHTING_SETUP									\
	DO_VS_SPOT_SHADOW_SETUP										\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 1 )			\
	SECOND_LIGHT_SPOT_VS_SHADOW_CALC					\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 2 )			\
	THIRD_LIGHT_SPOT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )	\
	DO_PS_LIGHTING_SETUP																				\
	DO_PS_POINT_LIGHTING_SETUP																	\
	DO_PS_POINT_SHADOW_SETUP																		\
	DO_PS_LIGHT_CALCULATIONS																		\
	FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 0 )			\
	DO_PS_SPOT_LIGHTING_SETUP																		\
	DO_PS_SPOT_SHADOW_SETUP																			\
	DO_PS_LIGHT_CALCULATIONS																		\
	SECOND_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 1 )			\
	DO_PS_LIGHT_CALCULATIONS																		\
	THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 2 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )


#elif defined( _LIGHTING_PPS_ )

//
// Two point + one spot light
//

#define DECLARE_DIRECT_LIGHTING_PARAMS	\
	DECLARE_POINT_PARAMS( 0 )							\
	DECLARE_POINT_PARAMS( 1 )							\
	DECLARE_SPOT_PARAMS( 2 )							\
	FIRST_LIGHT_POINT_SHADOW_PARAMS				\
	SECOND_LIGHT_POINT_SHADOW_PARAMS			\
	THIRD_LIGHT_SPOT_SHADOW_PARAMS

#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )	\
	DECLARE_POINT_INTERPOLATORS( __index__, 0 )								\
	FIRST_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )				\
	DECLARE_POINT_INTERPOLATORS( __index__, 1 )								\
	SECOND_LIGHT_POINT_SHADOW_INTERPOLATORS( __index__ )			\
	DECLARE_SPOT_INTERPOLATORS( __index__, 2 )								\
	THIRD_LIGHT_SPOT_SHADOW_INTERPOLATORS( __index__ )

#define DO_VS_DIRECT_LIGHTING_CALCULATIONS	\
	DO_VS_POINT_LIGHTING_SETUP								\
	DO_VS_POINT_SHADOW_SETUP									\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 0 )		\
	FIRST_LIGHT_POINT_VS_SHADOW_CALC					\
	DO_VS_POINT_LIGHTING_CALCULATIONS( 1 )		\
	SECOND_LIGHT_POINT_VS_SHADOW_CALC					\
	DO_VS_SPOT_LIGHTING_SETUP									\
	DO_VS_SPOT_SHADOW_SETUP										\
	DO_VS_SPOT_LIGHTING_CALCULATIONS( 2 )			\
	THIRD_LIGHT_SPOT_VS_SHADOW_CALC

#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )	\
	DO_PS_LIGHTING_SETUP																				\
	DO_PS_POINT_LIGHTING_SETUP																	\
	DO_PS_POINT_SHADOW_SETUP																		\
	DO_PS_LIGHT_CALCULATIONS																		\
	FIRST_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 0 )			\
	DO_PS_LIGHT_CALCULATIONS																		\
	SECOND_LIGHT_POINT_PS_SHADOW_CALC( __eye__ )														\
	DO_PS_POINT_LIGHTING_CALCULATIONS( __accumulator__, 1 )			\
	DO_PS_SPOT_LIGHTING_SETUP																		\
	DO_PS_SPOT_SHADOW_SETUP																			\
	DO_PS_LIGHT_CALCULATIONS																		\
	THIRD_LIGHT_SPOT_PS_SHADOW_CALC( __eye__ )															\
	DO_PS_SPOT_LIGHTING_CALCULATIONS( __accumulator__, 2 )

#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )

#else

//
// Fallback "no direct lighting" version
//
#define DECLARE_DIRECT_LIGHTING_PARAMS
#define DECLARE_DIRECT_LIGHTING_INTERPOLATORS( __index__ )
#define DO_VS_DIRECT_LIGHTING_CALCULATIONS
#define DO_PS_DIRECT_LIGHTING_CALCULATIONS( __accumulator__, __eye__ )	\
	DO_PS_LIGHTING_SETUP																									\
	DO_PS_LIGHT_CALCULATIONS
#define DO_VERTEXLIGHTING_DIRECT_LIGHTING_CALCULATIONS( __worldPos__, __worldNormal__, __accumulator__ )

#endif

#endif	// defined( _3DSMAX_ )


#endif	//  _DIRECT_LIGHTING_GLOBALS_FXH
